/*
*	Author:				Niklas Menke - https://www.niklas-menke.de/ (GE)
*	Description:		Some gas meters have a magnet in the meter mechanism. This device monitors the magnet field of this magnet to calculate the meter reading.
*						The controller provide a RS-485 interface with the Modbus RTU protocol. So a user can query the meter reading for furthur use.
*						The device address can be set with coding switches and the baud rate with a pin header.
*						A small LED is intended to show the state of this device.
*						More Information: https://www.niklas-menke.de/projekte/gaszaehler-auslesen/ (GE)
*
*	File:				main.c
*	Description:		Contains the main function and global ISRs
*	Microcontroller:	Attiny861A@8MHz; No other microcontrollers were tested
*	Date (created):		Mar 9, 2021
*
*	Version (Hardware):	2.2.0
*	Version (Software):	1.1.0
*	Date (updated):		Jan 16, 2022
*	Change log:			1.0.0:
*							- First release
*						1.1.0:
*							- Set clock prescaler automatically to 0 (8 MHz)
*							- Optimized pin change interrupts
*	ToDo:				- (Ideas or Problems? --> https://www.niklas-menke.de/kontakt/ (GE))
*
*
*	WARNING!:	This code was only tested with the Attiny861A controller and a CPU frequency of 8MHz. Support is not guaranteed for other conditions!
*/



// ----- Include needed files BEGIN -----
#include <avr/interrupt.h>
#include <avr/io.h>
#include <util/delay.h> 
#include "usi_uart.h"
#include "modbus_rtu_slave.h"
// ----- Include needed files END -----



// ----- Global Variables BEGIN -----
static uint8_t lps_sensor, lps_data, lps_settings;	// Last pin states of sensor, DI (UART), address and baudrate pins
// ----- Global Variables END -----


// ----- Main function BEGIN -----
int main(void)
{
	// Initialization
	CLKPR = 0x80;		// Clock prescaler will be changed
	CLKPR = 0x00;		// Set clock prescaler to 0 (8 MHz)
	DDRA = 0x02;		// Set TXE/!RXE as output
	DDRB = 0x06;		// Set DO and LED as output
	PORTA = 0xfd;		// Enable pullups of ADDRESS and BAUDRATE
	PORTB = 0x6b;		// Enable pullups of DI and ADDRESS. Set DO to high level
	PRR = 0x0f;			// Shut down Timer0, Timer1, USI and ADC module
	GIMSK = 0x30;		// Enable pin change interrupts globally
	PCMSK0 = 0xc0;		// Enable pin change interrupts for BAUDRATE pins
	PCMSK1 = 0x58;		// Enable pin change interrupts for ADDRESS and SENSOR
	uart_TimerConfig();	// Set setpoint and prescaler of the UART timer
	rtu_setAddress();	// Get device address from the coding switches
	
	// Show that the device is ready
	for(uint8_t i = 0; i < 6; i++) {
		PORTB ^= (1<<PB2);
		_delay_ms(500);
	}
	
	// Get current pin states
	lps_sensor = PINB&(1<<PB4);						// Magnetoresistive sensor
	lps_data = PINB&(1<<PB0);						// DI pin of the UART
	lps_settings = (PINA&0xc0) | ((PINB&0x48)>>1);	// Address and baudrate pins
	
	sei();			// Enable interrupts globally
	uart_listen();	// Monitoring DI pin for incoming bytes
	
	// Main loop
	while(1) {
		if(uart_available()) {								// At least one byte was received
			uint8_t bytes_recevied = 0;						// Save the last number of received bytes
			while(bytes_recevied != uart_available()) {		// Wait until all bytes are received
				bytes_recevied = uart_available();			// Save the last number of received bytes
				rtu_break(1.5);								// Wait 1.5 symbols to receive a possible new byte
			}
			rtu_break(2);	// Wait to guaranteed a break of 3.5 symbols between the telegrams
			rtu_response();	// Check the received bytes and send a response if necessary
		}
		rtu_break(1.5);
	}
}
// ----- Main function END -----



// ----- Pin change interrupt BEGIN -----
ISR(PCINT_vect) {	
	// DI pin of the UART interface
	if((PINB&(1<<PB0)) != lps_data) {
		lps_data = PINB&(1<<PB0);											// Save new pin state
		if(!lps_data && (uart_status() == USI_UART_READY)) uart_receive();	// Receive the incoming byte
	}
	
	// The magnetoresistive sensor was triggered
	if((PINB&0x10) != lps_sensor) {
		lps_sensor = (PINB&0x10);	// Save new pin state
		if(lps_sensor) {			// Count meter reading when there is a positive flag
			PORTB |= (1<<PB2);
			if(rtu_read(1) < UINT16_MAX) rtu_write(1, rtu_read(1)+1);
			else {	// Overflow
				rtu_write(0, rtu_read(0)+1);
				rtu_write(1,0);
			}
			PORTB &= ~(1<<PB2);
		}	
	}
	
	// Device address or baud rate was changed
	if(((PINA&0xc0) | ((PINB&0x48)>>1)) != lps_settings) {
		lps_settings = (PINA&0xc0) | ((PINB&0x48)>>1);	// Save new pin state
		PRR &= ~(1<<PRTIM1);							// Start Timer1 module
		TCNT1 = 0;										// Reset counter of Timer1
		TIMSK &= 0x1b;									// Disable all interrupts of Timer1
		TIMSK |= (1<<TOIE1);							// Enable overflow interrupt
		TCCR1B = 0x0f;									// Enable Timer1 with 16384 prescaler --> Overflow after ~0,5s@8MHz
	}
}
// ----- Pin change interrupt END -----



// ----- Timer1 overflow interrupt BEGIN -----
ISR(TIMER1_OVF_vect) {
	PORTB |= (1<<PB2);		// Show update process
	TCCR1B = 0;				// Disable Timer1
	TIMSK &= ~(1<<TOIE1);	// Disable overflow interrupt
	PRR |= (1<<PRTIM1);		// Shut down Timer1 module
	uart_TimerConfig();		// Calculate new prescaler and setpoint of the UART timer
	rtu_setAddress();		// Get the new device address from the coding switches
	PORTB &= ~(1<<PB2);		// Disable LED
}
// ----- Timer1 overflow interrupt END -----