/*
*	Author:				Niklas Menke - https://www.niklas-menke.de/ (GE)
*	Description:		This library uses the USI (Universal Serial Interface) to provide a UART. Also there are some supportive routines.
*
*	File:				usi-uart.c
*	Description:		Functions definition
*	Microcontroller:	Attiny861A@8MHz; No other microcontrollers were tested
*	Date (created):		Mar 10, 2021
*
*	Version (Hardware):	-
*	Version (Software):	1.0.0
*	Date (updated):		-
*	Change log:			-
*	ToDo:				- (Ideas or Problems? --> https://www.niklas-menke.de/kontakt/ (GE))
*
*
*	WARNING!:	This code was only test with the Attiny861A controller and a CPU frequency of 8MHz. Support is not guaranteed for other conditions!
*				To get a CPU frequency of 8MHz, you have to clear the clock divider fuse (LOW.CKDIV8).
*/

#include "usi_uart.h"

enum UART_STATE{NOTINIT, AVAILABLE, BREAK, RECEIVING, FIRST, SECOND};	// Status code of the UART
static volatile enum UART_STATE UART_RX = NOTINIT;	// Status of Rx mode
static volatile enum UART_STATE UART_TX = NOTINIT;	// Status of Tx mode
static uint8_t prescaler, setpoint;					// Storage for prescaler configuration and setpoint of Timer0
static uint8_t data_rx[16];							// Buffer for received bytes
static uint8_t data_avail_rx = 0;					// Number of available bytes in Rx buffer
static uint8_t data_tx;								// Byte which is to be transmitted


// ----- Monitor DI pin for incoming bytes BEGIN -----
void uart_listen(void) {
	while(UART_RX == RECEIVING || UART_TX == FIRST || UART_TX == SECOND);	// Wait until receiving or transmitting is finished
	DDRA |= (1<<PA1);		// Define TXE/!RXE as output
	PORTA &= ~(1<<PA1);		// Enable Rx mode
	//DDRB &= ~(1<<PB0);	// Define DI as input
	//DDRB |= (1<<PB1);		// Define DO as output
	DDRB &= 0xfc;			// Define DI and DO as input
	PORTB |= 0x03;			// Enable pullup for DI and set DO to high level
	GIMSK |= (1<<PCIE0);	// Enable pin change interrupts globally
	PCMSK1 |= (1<<PCINT8);	// Enable pin change interrupt of DI
	UART_RX = AVAILABLE;	// UART is available to receive bytes
	UART_TX = NOTINIT;		// UART is not available to transmit bytes
}
// ----- Monitor DI pin for incoming bytes END -----



// ----- Stop monitoring DI pin for incoming bytes BEGIN -----
void uart_stop(void) {
	while(UART_RX == RECEIVING || UART_TX == FIRST || UART_TX == SECOND);	// Wait until receiving or transmitting is finished
	PCMSK1 &= ~(1<<PCINT8);	// Disable pin change interrupt of DI
	DDRA |= (1<<PA1);		// Define TXE/!RXE as output
	PORTA &= ~(1<<PA1);		// Enable Rx mode
	DDRB &= ~(1<<PB0);		// Define DI as input
	DDRB |= (1<<PB1);		// Define DO as output
	PORTB |= 0x03;			// Enable pullup for DI and set DO to high level
	UART_RX = NOTINIT;		// UART is not available to receive bytes
}
// ----- Stop monitoring DI pin for incoming bytes END -----



// ----- Return baud rate from the pin header BEGIN -----
uint16_t uart_getBaudrate(void) {
	DDRA &= 0x3f;	// Set BAUDRATE pins as input
	PORTA |= 0xc0;	// Enable pullups of BAUDRATE pins
	return ((PINA&0xc0)>>6) == 2 ? 19200 : ((PINA&0xc0)>>6) == 1 ? 38400 : 9600;	// Return baud rate
}
// ----- Return baud rate from the pin header END -----



// ----- Calculate timer setpoint and prescaler BEGIN -----
void uart_TimerConfig(void) {
	uint16_t buffer = F_CPU/uart_getBaudrate()/254;	// Minimum value for prescaler
	if(!buffer) {				// No prescaler
		prescaler = 1;			// Register value 1
		buffer = 1;
	}
	else if(buffer < 8) {		// 8-prescaler
		prescaler = 2;			// Register value 2
		buffer = 8;
	}
	else if(buffer < 64) {		// 64-prescaler
		prescaler = 3;			// Register value 3
		buffer = 64;
	}
	else if(buffer < 256) {		// 256-prescaler
		prescaler = 4;			// Register value 4
		buffer = 256;
	}
	else if(buffer < 1024) {	// 1024-prescaler
		prescaler = 5;			// Register value 5
		buffer = 1024;
	}
	else prescaler = 0;			// CPU frequency too high or baud rate too low
	
	if(prescaler) setpoint = F_CPU/uart_getBaudrate()/buffer+1;	// If prescaler has a legal value, then calculate setpoint for Timer0
	else setpoint = 0;											// CPU frequency too high or baud rate too low
}
// ----- Calculate timer setpoint and prescaler END -----



// ----- Receive one byte BEGIN -----
void uart_receive(void) {
	if(UART_RX == AVAILABLE) {	// Only if Rx mode is available
		cli();					// Disable interrupts globally during configuration
		PCMSK1 &= ~(1<<PCINT8);	// Disable pin change interrupt of DI 
		PRR &= 0x09;			// Start Timer0 and USI module
		
		// Timer0
		TIMSK &= 0xe4;		// Disable all interrupts
		TCCR0A = 0x01;		// Enable CTC-mode
		TCNT0L = 0;			// Reset counter
		OCR0A = setpoint;	// Set setpoint
		TCCR0B = prescaler;	// Enable timer
		
		// USI
		USISR = 0x48;	// Clear overflow flag and set counter value to 8 of the 4-bit-timer
		USICR = 0x54;	// Enable 4-bit-counter overflow interrupt, set 3-wire-mode and select Timer0 compare match as clock
		
		UART_RX = RECEIVING;	// Receiving byte
		PORTB |= (1<<PB2);		// Show receiving process
		sei();					// Enable interrupts globally
	}
}
// ----- Receive one byte END -----



// ----- Transmit one byte BEGIN -----				
void uart_transmit(const uint8_t byte) {
	while(UART_RX == RECEIVING || UART_TX == FIRST || UART_TX == SECOND);	// Wait until receiving or transmitting is finished
	cli();										// Disable interrupts globally during configuration
	PRR &= 0x09;								// Start Timer0 and USI module
	if(UART_RX == AVAILABLE) UART_RX = BREAK;	// UART is not available to receive bytes
	PCMSK1 &= ~(1<<PCINT8);						// Disable pin change interrupt of DI
	DDRA |= (1<<PA1);							// Define TXE/!RXE as output
	PORTA &= ~(1<<PA1);							// Enable Rx mode
	DDRB &= ~(1<<PB0);							// Define DI as input
	DDRB |= (1<<PB1);							// Define DO as output
	PORTB |= 0x07;								// Enable pullup for DI, set DO to high level and turn on LED
	
	data_tx = rev_byte(byte);	// Save byte into buffer with LSB first
	
	// USI
	USISR = 0x48;				// Clear overflow flag and set counter value to 8 of the 4-bit-timer
	USICR = 0x54;				// Enable 4-bit-counter overflow interrupt, set 3-wire-mode and select Timer0 compare match as clock
	USIDR = 0 | (data_tx>>1);	// Save the start bit and the first seven data bits into the USI data register
	
	// Timer0
	TIMSK &= 0xe4;		// Disable all interrupts
	TCCR0A = 0x01;		// Enable CTC-mode
	TCNT0L = 0;			// Reset counter
	OCR0A = setpoint;	// Set setpoint
	TCCR0B = prescaler;	// Enable timer
	
	PORTA |= (1<<PA1);	// Enable Tx mode
	UART_TX = FIRST;	// Transmitting the first byte
	sei();				// Enable interrupts globally
}
// ----- Transmit one byte END -----



// ----- Transmit an array of bytes BEGIN -----
void uart_transmitArray(const uint8_t *byte, const uint8_t length) {
	for(uint8_t i = 0; i < length; i++) uart_transmit(byte[i]);
}
// ----- Transmit an array of bytes END -----



// ----- Return number of received bytes in buffer BEGIN -----
uint8_t uart_available(void) {
	return data_avail_rx;
}
// ----- Return number of received bytes in buffer END -----



// ----- Return first byte from buffer without delete them from buffer BEGIN -----
uint8_t uart_readByte(void) {
	return data_rx[0];
}
// ----- Return first byte from buffer without delete them from buffer END -----



// ----- Return first byte from buffer and delete them from buffer BEGIN -----
uint8_t uart_getByte(void) {
	if(data_avail_rx) {
		uint8_t buffer = data_rx[0];											// Save first byte
		for(uint8_t i = 0; i < data_avail_rx-1; i++) data_rx[i] = data_rx[i+1];	// Move all other bytes one up
		data_avail_rx--;
		return buffer;
	}
	return 0;
}
// ----- Return first byte from buffer and delete them from buffer END -----



// ----- Clear buffer BEGIN -----
void uart_delete(void) {
	data_avail_rx = 0;
}
// ----- Clear buffer BEGIN -----



// ----- Reverse bits of a byte BEGIN -----
uint8_t rev_byte(uint8_t rbyte) {
	rbyte = ((rbyte >> 1) & 0x55) | ((rbyte << 1) & 0xaa);
	rbyte = ((rbyte >> 2) & 0x33) | ((rbyte << 2) & 0xcc);
	rbyte = ((rbyte >> 4) & 0x0f) | ((rbyte << 4) & 0xf0);
	return rbyte;
}
// ----- Reverse bits of a byte END -----



// ----- USI overflow interrupt BEGIN -----
ISR(USI_OVF_vect) {
	if(UART_RX == RECEIVING) {	// Full byte was received
		TCCR0B = 0;		// Disable Timer0
		USICR = 0;		// Disable USI
		USISR = 0x40;	// Clear overflow interrupt flag of 4-bit-timer
		
		data_rx[data_avail_rx++] = rev_byte(USIDR);	// Save received byte in buffer
		if(data_avail_rx > 15) data_avail_rx = 0;	// Buffer is full --> Data will be overwritten next time
		
		PRR |= 0x06;			// Shut down Timer0 and USI module
		PCMSK1 |= (1<<PCINT8);	// Enable pin change interrupt of DI
		UART_RX = AVAILABLE;	// UART is available to receive next byte
		PORTB &= ~(1<<PB2);		// Turn off LED
	}
	else if(UART_TX == FIRST) {			// First byte was transmitted
		USIDR = (data_tx<<7) | 0x7f;	// Save last data bit and into the USI data register and fill the rest with 1
		USISR = 0x4d;					// Clear overflow flag and set counter value to 13 of the 4-bit-timer
		UART_TX = SECOND;				// Transmitting second byte
	}
	else if(UART_TX == SECOND) {
		PORTA &= ~(1<<PA1);					// Disable Tx mode
		TCCR0B = 0;							// Disable Timer0
		USICR = 0;							// Disable USI
		USISR = 0x40;						// Clear overflow flag of 4-bit-timer
		PRR |= 0x06;						// Shut down Timer0 and USI module
		UART_TX = NOTINIT;					// Transmitting is finished
		DDRB &= ~(1<<PB1);					// Define DO as input
		PORTB &= ~(1<<PB2);					// Turn off LED
		if(UART_RX == BREAK) uart_listen();	// Monitor DI pin for incoming bytes
	}
}
// ----- USI overflow interrupt END -----